#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# notiplus.py
# NOTIFICADOR SELECTIVO de noticias de Investing ES con notificaciones y FILTRO por palabras
#
# Requisitos:
#   py -m pip install playwright beautifulsoup4 lxml winotify
#   py -m playwright install
#
# Uso:
#   py notiplus.py <minutos> [palabra1] [palabra2] [palabra3] [palabra4] [palabra5]
#   Ejemplo:
#   py notiplus.py 5 adqui compra contrato acuerdo firma
#
# Notas:
# - Si <minutos> no se indica o es inválido, se usa 5 por defecto.
# - Se aceptan hasta 5 palabras de filtro. Si no pasas palabras, no se filtra (se muestran todas).
# - En consola se muestra primero el enlace de la noticia y a continuación el resumen.

import asyncio, json, sys, time, os, shutil, re, unicodedata
from pathlib import Path
from urllib.parse import urljoin
from typing import List, Tuple
from bs4 import BeautifulSoup

SEED_URL = "https://es.investing.com/news"
DATA_DIR = Path(__file__).parent
SEEN_FILE = DATA_DIR / "seen_investing_plus.json"   # separado de notinoti.py
LOG_FILE  = DATA_DIR / "investing_news_plus.log"

CHECK_INTERVAL_SECONDS = 300      # 5 minutos por defecto
TOAST_ON_NO_NEWS = False          # True para avisar aunque no haya novedades
TOAST_MAX_LINES = 3               # Max titulares/resúmenes en la notificación
TOAST_MAX_CHARS = 300             # Max longitud del texto del toast
KEYWORDS_MAX = 5                  # Máximo de palabras de filtro

# -------------------- Utilidades --------------------

def clear_console():
    """Limpia la consola"""
    os.system('cls' if os.name == 'nt' else 'clear')

def normalize_text(s: str) -> str:
    """Minúsculas y sin acentos para comparar."""
    s = s.lower()
    s = unicodedata.normalize("NFD", s)
    s = "".join(c for c in s if unicodedata.category(c) != "Mn")
    return s

def show_header(refresh_minutes: int, keywords: List[str]):
    """Muestra la cabecera ASCII con refresco y palabras de filtro."""
    kw_text = ", ".join(keywords) if keywords else "(sin filtro)"
    header = f"""
  ╔═══════════════════════════════════════════════════════════════════════╗
  ║                                                                       ║
  ║███╗   ██╗  ██████╗  ████████╗ ██╗ ██████╗  ██╗      ██╗   ██╗  █████╗ ║
  ║████╗  ██║ ██╔═══██╗ ╚══██╔══╝ ██║ ██╔══██╗ ██║      ██║   ██║ ██╔═══╝ ║
  ║██╔██╗ ██║ ██║   ██║    ██║    ██║ ██████╔╝ ██║      ██║   ██║ ███████╗║ 
  ║██║╚██╗██║ ██║   ██║    ██║    ██║ ██╔═══╝  ██║      ██║   ██║ ╚════██║║ 
  ║██║ ╚████║ ╚██████╔╝    ██║    ██║ ██║      ███████╗ ╚██████╔╝ ███████║║ 
  ║╚═╝  ╚═══╝  ╚═════╝     ╚═╝    ╚═╝ ╚═╝      ╚══════╝  ╚═════╝  ╚══════╝║ 
  ║                                                                       ║
  ║           Notificador de Noticias PLUS (investing.com)                ║
  ║                Filtro por palabras en texto-resumen                   ║
  ║                                                                       ║
  ║  By Rafa Lome ©© 2025  -  https://calentamientoglobalacelerado.net    ║
  ╚═══════════════════════════════════════════════════════════════════════╝
     Refresco (min): {refresh_minutes:<3}  |  Palabras: {kw_text:<36}
  ═════════════════════════════════════════════════════════════════════════
    """
    print(header)

def get_separator():
    """Devuelve una línea separadora de asteriscos de la mitad del ancho de la consola"""
    try:
        console_width = shutil.get_terminal_size().columns
        separator_width = console_width // 2
        return "*" * separator_width
    except:
        return "*" * 40  # Fallback si no se puede obtener el ancho

def load_seen() -> set:
    if SEEN_FILE.exists():
        try:
            return set(json.loads(SEEN_FILE.read_text(encoding="utf-8")))
        except Exception:
            return set()
    return set()

def save_seen(seen: set) -> None:
    SEEN_FILE.write_text(json.dumps(sorted(seen), ensure_ascii=False), encoding="utf-8")

def log(msg: str) -> None:
    ts = time.strftime("%Y-%m-%d %H:%M:%S")
    line = f"[{ts}] {msg}"
    print(line, flush=True)
    with LOG_FILE.open("a", encoding="utf-8") as f:
        f.write(line + "\n")

def log_block(title: str, lines: List[str]) -> None:
    log(title)
    for ln in lines:
        log(f"  - {ln}")

# -------------------- Notificaciones --------------------

def notify(title: str, message: str) -> None:
    """
    Notificación nativa en Windows 10/11 mediante winotify (WinRT).
    """
    try:
        from winotify import Notification, audio
        msg = (message or "").strip().replace("\r", "")
        if len(msg) > TOAST_MAX_CHARS:
            msg = msg[:TOAST_MAX_CHARS - 3] + "..."
        toast = Notification(app_id="Investing Watch+",
                             title=title,
                             msg=msg,
                             duration="short")
        toast.set_audio(audio.Default, loop=False)
        toast.show()
    except Exception as e:
        log(f"NOTIFICACION NO ENVIADA: {e}")

# -------------------- Descarga y parsing --------------------

async def fetch_html_playwright(url: str) -> str:
    from playwright.async_api import async_playwright
    async with async_playwright() as p:
        browser = await p.chromium.launch(headless=True)
        context = await browser.new_context(
            user_agent=("Mozilla/5.0 (Windows NT 10.0; Win64; x64) "
                        "AppleWebKit/537.36 (KHTML, like Gecko) "
                        "Chrome/124.0.0.0 Safari/537.36"),
            locale="es-ES"
        )
        page = await context.new_page()
        resp = await page.goto(url, wait_until="domcontentloaded", timeout=60000)
        if not resp or not resp.ok:
            await browser.close()
            raise RuntimeError(f"HTTP error: {resp.status if resp else 'sin respuesta'}")
        try:
            await page.wait_for_selector("article, a[href*='/news/']", timeout=15000)
        except Exception:
            pass
        html = await page.content()
        await browser.close()
        return html

def _extract_summary_from_article(art) -> str:
    """
    Intenta extraer un texto-resumen desde la tarjeta/listado:
    - primer <p>
    - o div con clase que sugiera 'summary'/'text'/'content'
    - fallback: texto del enlace (título)
    """
    # 1) primer párrafo directo
    p = art.find("p")
    if p and p.get_text(strip=True):
        return p.get_text(" ", strip=True)

    # 2) divs con clases típicas
    for div in art.select("div"):
        cls = " ".join(div.get("class", [])).lower()
        if any(x in cls for x in ("summary", "resumen", "text", "content", "desc")):
            txt = div.get_text(" ", strip=True)
            if txt:
                return txt

    # 3) fallback: texto completo del artículo sin el título/enlace
    txt = art.get_text(" ", strip=True) or ""
    return txt

def parse_items(html: str) -> List[Tuple[str, str, str]]:
    """
    Devuelve lista de tuplas: (titulo, url, resumen)
    """
    soup = BeautifulSoup(html, "lxml")
    items: List[Tuple[str, str, str]] = []

    # Heurística 1: artículos con enlace /news/
    for art in soup.select("article"):
        a = art.select_one("a[href*='/news/']")
        if not a:
            continue
        title = (a.get_text(strip=True) or "").strip()
        href = a.get("href")
        if not href:
            continue
        url = urljoin(SEED_URL, href)
        summary = _extract_summary_from_article(art) or title
        items.append((title, url, summary))

    # Respaldo: enlaces sueltos
    if not items:
        for a in soup.select("a[href*='/news/']"):
            title = (a.get_text(strip=True) or "").strip()
            href  = a.get("href")
            if title and href:
                url = urljoin(SEED_URL, href)
                items.append((title, url, title))  # sin resumen en listado

    # Deduplicar manteniendo orden por URL
    seen_urls = set()
    dedup: List[Tuple[str, str, str]] = []
    for t, u, s in items:
        if u in seen_urls:
            continue
        seen_urls.add(u)
        dedup.append((t, u, s))
    return dedup

# -------------------- Resumen / Filtro --------------------

def filter_by_keywords(items: List[Tuple[str, str, str]], keywords: List[str]) -> List[Tuple[str, str, str]]:
    """
    Filtra por aparición de cualquiera de las 'keywords' en el texto-resumen.
    Si no hay keywords, no filtra (devuelve todo).
    Coincidencia por subcadena, insensible a acentos y mayúsculas.
    """
    if not keywords:
        return items

    norm_keys = [normalize_text(w) for w in keywords if w]
    out = []
    for title, url, summary in items:
        norm_sum = normalize_text(summary)
        if any(k in norm_sum for k in norm_keys):
            out.append((title, url, summary))
    return out

def build_toast_message(new_items: List[Tuple[str, str, str]]) -> str:
    if not new_items:
        return "Sin noticias nuevas."
    # Para el toast usamos el resumen recortado
    lines = []
    for _title, _url, summary in new_items[:TOAST_MAX_LINES]:
        s = summary.strip().replace("\r", "").replace("\n", " ")
        if len(s) > 120:
            s = s[:117] + "..."
        lines.append(f"- {s}")
    return "\n".join(lines)

# -------------------- Ejecución --------------------

async def run_once(keywords: List[str]) -> None:
    # Mostrar separador antes de cada verificación
    print(get_separator())

    html = await fetch_html_playwright(SEED_URL)
    items = parse_items(html)
    items = filter_by_keywords(items, keywords)

    seen = load_seen()
    new_items: List[Tuple[str, str, str]] = []
    for title, url, summary in items:
        if url not in seen:
            new_items.append((title, url, summary))
            # En consola: primero el enlace y después el resumen (exigido)
            out_summary = summary.strip().replace("\n", " ")
            log(f"{url} | {out_summary}")
            seen.add(url)

    if new_items:
        # Resumen a consola/log
        lines = [f"{i+1}. {u}" for i, (_t, u, _s) in enumerate(new_items[:5])]
        log_block("NUEVAS (filtradas):", lines)
        # Toast
        notify(
            title=f"Investing ES+ - {len(new_items)} noticia(s) nueva(s)",
            message=build_toast_message(new_items)
        )
    else:
        log("Sin noticias nuevas (tras aplicar filtro).")
        if TOAST_ON_NO_NEWS:
            notify("Investing ES+", "Sin noticias nuevas (filtro activo).")

    save_seen(seen)

async def main_loop(every_seconds: int, keywords: List[str]) -> None:
    log(f"Arrancando monitor PLUS (cada {every_seconds//60} min) | Filtro: {', '.join(keywords) if keywords else '(sin filtro)'}")
    while True:
        try:
            await run_once(keywords)
        except Exception as e:
            log(f"ERROR: {e} | Ejemplo de uso: py notiplus.py 5 adqui compra contrato acuerdo firma")
        await asyncio.sleep(every_seconds)

def parse_cli_args(argv: List[str]) -> Tuple[int, List[str]]:
    """
    Interpreta argumentos:
    - argv[1]: minutos (int). Si falta o es inválido -> 5.
    - argv[2:]: hasta 5 palabras. Si hay más, se ignoran y se avisa.
    - Si argv[1] no es int, se asume que no se han indicado minutos y se usa 5,
      usando argv[1:] como palabras.
    """
    minutes = 5
    keywords: List[str] = []

    if len(argv) <= 1:
        # Sin argumentos: minutos por defecto, sin filtro
        return minutes, keywords

    # Intentar minutos en argv[1]
    try:
        m = int(argv[1])
        if m <= 0:
            print("Error: el intervalo debe ser un número positivo. Usando valor por defecto (5 min).")
        else:
            minutes = m
        # Palabras desde argv[2:]
        keywords = argv[2:]
    except ValueError:
        # argv[1] no es int -> minutos por defecto y palabras desde argv[1:]
        print("Aviso: no se indicó un número válido de minutos. Se usará 5 por defecto.")
        print("Ejemplo: py notiplus.py 5 adqui compra contrato acuerdo firma")
        keywords = argv[1:]

    if len(keywords) > KEYWORDS_MAX:
        print(f"Aviso: has indicado más de {KEYWORDS_MAX} palabras. Solo se tendrán en cuenta las primeras {KEYWORDS_MAX}.")
        keywords = keywords[:KEYWORDS_MAX]

    # Limpiar palabras vacías/blancos
    keywords = [w.strip() for w in keywords if w and w.strip()]
    return minutes, keywords

if __name__ == "__main__":
    # Parseo de CLI
    minutes, keywords = parse_cli_args(sys.argv)

    # Preparación de entorno
    clear_console()
    show_header(minutes, keywords)

    every_seconds = (minutes if minutes > 0 else 5) * 60
    if minutes > 0:
        print(f"Intervalo activo: {minutes} minuto(s)")
    else:
        print("Usando valor por defecto (5 min).")

    try:
        asyncio.run(main_loop(every_seconds, keywords))
    except KeyboardInterrupt:
        print(get_separator())
        log("Finalizado por usuario.")
        sys.exit(0)
